/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import org.nexuswob.gui.TableModel3;
import org.nexuswob.util.Cell;
import org.nexuswob.util.Util.Record;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.protocol.Benutzerstatus;
import net.javacomm.transfer.TransferUser;



@SuppressWarnings("serial")
public class JScrollBenutzerantraege extends JScrollPane implements Babelfish {

  public final static int MAIL = 0;
  public final static int BENUTZERSTATUS = 1;
  private JTable table = new JTable();
  private TableModel3 model3 = new TableModel3() {

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
      return columnIndex == BENUTZERSTATUS;
    }

  };
  private Map<String, String> map = new HashMap<>();
  private TableColumnModel columnModel;
  private JTableHeader header;
  private JComboBox<Benutzerstatus> combobox = new JComboBox<>(
      new Benutzerstatus[] {Benutzerstatus.MÖCHTE_VOLLMITGLIED_WERDEN,
          Benutzerstatus.VOLLMITGLIEDSCHAFT_BESTÄTIGT, Benutzerstatus.MITGLIEDSCHAFT_ABGELEHNT}
  );
  private MultilingualString benutzerstatus = new MultilingualString(KEY.STRING_BENUTZERSTATUS);
  private MultilingualString correo = new MultilingualString(KEY.STRING_MAIL);
  private DefaultTableCellRenderer headerRenderer;
  private MultilingualString moechteMitgliedWerden = new MultilingualString(
      KEY.STRING_MOECHTE_MITGLIED_WERDEN
  );
  private MultilingualString abgelehnt = new MultilingualString(KEY.STRING_MITGLIEDSCHAFT_ABGELEHNT);
  private MultilingualString confirmar = new MultilingualString(KEY.STRING_MITGLIEDSCHAFT_BESTAETIGT);

  public JScrollBenutzerantraege() {

    model3.setHeader(correo.toString(), benutzerstatus.toString());
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setModel(model3);
    table.setRowHeight(26);
    table.setFillsViewportHeight(true);
    table.setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);

    centerHeader();

    columnModel = table.getColumnModel();
    columnModel.getColumn(MAIL).setPreferredWidth(200);
    columnModel.getColumn(BENUTZERSTATUS).setCellEditor(new DefaultCellEditor(combobox));

    this.getViewport().setOpaque(true);
    setViewportView(table);
    this.setOpaque(true);
    setBorder(null);

  }



  /**
   * Die Spaltenüberschriften werden zentriert.
   * 
   */
  public void centerHeader() {
    header = table.getTableHeader();
    headerRenderer = (DefaultTableCellRenderer) header.getDefaultRenderer();
    headerRenderer.setHorizontalAlignment(SwingConstants.CENTER);
  }



  /**
   * Zeige Kontoaktivitäten an.
   * 
   * @param userliste
   *                  diese User haben Kontoaktivitäten
   */
  public void setKontoaktivitaet(List<TransferUser> userliste) {
    model3.removeAll();
    map.clear();

    userliste.forEach((user) -> {

      // 1. Zelle
      Record record = new Record();
      map.put(user.getMail(), user.getUserid());

      // 2. Zelle
      Cell<String> mail = new Cell<>();
      mail.setValue(user.getMail());
      record.add(mail);

      // 3. Zelle
      Cell<String> benutzerstatus = new Cell<>();
      benutzerstatus.setValue(Benutzerstatus.toBenutzerstatus(user.getBenutzerstatus()).toString());
      record.add(benutzerstatus);

      model3.addRow(record);

    });

  }



  List<TransferUser> getKontoaktivitaet() {
    ArrayList<TransferUser> liste = new ArrayList<>();
    for (Record record : model3.getRecords()) {
      TransferUser transferuser = new TransferUser();

      transferuser.setMail((String) record.getCellAt(MAIL).getValue());
      transferuser.setUserid(map.get(transferuser.getMail()));

      // TOOD auf den richtigen Staus abbilden

      Benutzerstatus status = Benutzerstatus.valueOf((String) record.getCellAt(BENUTZERSTATUS).getValue());
      transferuser.setBenutzerstatus(status.getStatus());
      switch(status) {
        case VOLLMITGLIEDSCHAFT_BESTÄTIGT:
          liste.add(transferuser);
          break;
        case MITGLIEDSCHAFT_ABGELEHNT:
          liste.add(transferuser);
          break;
        default:
          break;
      }
    }
    return liste;
  }



  @Override
  public void setLanguage(ISO639 code) {
    benutzerstatus.setLanguage(code);
    correo.setLanguage(code);
    columnModel.getColumn(BENUTZERSTATUS).setHeaderValue(benutzerstatus.toString());
    columnModel.getColumn(MAIL).setHeaderValue(correo.toString());
    moechteMitgliedWerden.setLanguage(code);
    abgelehnt.setLanguage(code);
    confirmar.setLanguage(code);

  }

}
